/*
/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package com.inet.authentication.urlcookie;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletRequestWrapper;

/**
 * A sample that show how you can pass a cookie to the server as URL Parameter
 */
public class UrlCookie implements Filter {

    private static final String URL_COOKIE = "urlCookie";

    /**
     * {@inheritDoc}
     */
    @Override
    public void init( FilterConfig config ) throws ServletException {
        // nothing
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void doFilter( ServletRequest request, ServletResponse response, FilterChain chain ) throws IOException, ServletException {

        String urlCookie = request.getParameter( URL_COOKIE );
        if( urlCookie != null ) {
            int idx = urlCookie.indexOf( ':' );
            if( idx > 0 ) {
                final Cookie cookie = new Cookie( urlCookie.substring( 0, idx ), urlCookie.substring( idx + 1 ) );
                final HttpServletRequest req = (HttpServletRequest)request;

                // create a wrapper to pass the Principal because there is no setter
                request = new HttpServletRequestWrapper( req ) {
                    @Override
                    public Cookie[] getCookies() {
                        Cookie[] cookies = super.getCookies();
                        if( cookies == null ) {
                            return new Cookie[] { cookie };
                        }
                        ArrayList<Cookie> list = new ArrayList<>();
                        list.add( cookie );
                        list.addAll( Arrays.asList( cookies ) );
                        return list.toArray( new Cookie[list.size()] );
                    }

                    @SuppressWarnings( { "unchecked", "rawtypes" } )
                    @Override
                    public Enumeration getParameterNames() {
                        // remove the URL_COOKIE parameter from the parameter list
                        List list = Collections.list( super.getParameterNames() );
                        list.remove( URL_COOKIE );
                        return Collections.enumeration( list );
                    }
                };
            }
        }

        // call the i-net Clear Report servlet
        chain.doFilter( request, response );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void destroy() {
        // nothing
    }
}
